
// Neurosis Engine - LP23.com
// Copyright  Luigi Pino. All rights reserved.

/***************************************************************************/

#ifndef _NEUROSIS_ENGINE_MODEL_H_
#define _NEUROSIS_ENGINE_MODEL_H_

/***************************************************************************/

struct SRenderBatch {
	float	mTextureX, mTextureY;
	float	mColorR, mColorG, mColorB, mColorA;
	float	mNormalX, mNormalY, mNormalZ;
	float	mVertexX, mVertexY, mVertexZ;
};

/***************************************************************************/

class CNeurosisPoints {
	public:
		CNeurosisPoints();
		//! Constructor/Deconstructor
		~CNeurosisPoints();

		void	operator= (CNeurosisPoints &rhs);
		bool	operator== (CNeurosisPoints rhs);
		//! Operators
		bool	operator!= (CNeurosisPoints rhs);

		//! Initialize
		void	Initialize();

		color3	mColor;
		float3	mPosition;
};

/***************************************************************************/

class CNeurosisTriangles {
	public:
		CNeurosisTriangles();
		//! Constructor/Deconstructor
		~CNeurosisTriangles();

		void	operator= (CNeurosisTriangles &rhs);
		bool	operator== (CNeurosisTriangles rhs);
		//! Operators
		bool	operator!= (CNeurosisTriangles rhs);

		//! Initialize
		void	Initialize();

		float		mTransparency;														// Transparency [0.0f..1.0f]
		float3	mCenter;																	// Center
		float3	mNormal;																	// Normal
		float3	mSideNormals[3];													// Normal of corresponding sides
		float3	mPercent[3];															// Texture coordinates
		int			iMesh;																		// Mesh index
		int			iTexture;																	// Texture index
		int			mPoint[3];																// CNeurosisPoints index reference
};

/***************************************************************************/

class CNeurosisModel {
	public:
		CNeurosisModel();
		//! Constructor/Deconstructor
		~CNeurosisModel();

		//! Operators
		void	operator= (CNeurosisModel &rhs);

		//! Add point to model [checks duplicate points by default]
		int			Add_Point(float3 *newPoint, bool duplicateCheck = true);
		//! Add triangle to model
		int			Add_Triangle(int point0, int point1, int point2);
		//! Add triangle (from parameters passed during Render_Triangle)
		void		Add_Triangle(int iTexture, float bottomLeftX, float bottomLeftY, float bottomLeftZ, float topRightX, float topRightY, float topRightZ, float topLeftX, float topLeftY, float topLeftZ, float bottomLeftPercentX = 0.0f, float bottomLeftPercentY = 0.0f, float topRightPercentX = 1.0f, float topRightPercentY = 1.0f, float topLeftPercentX = 0.0f, float topLeftPercentY = 1.0f);
		//! Add quad (from parameters passed during Render_Quad)
		void		Add_Quad(int iTexture, float bottomLeftX, float bottomLeftY, float bottomLeftZ, float bottomRightX, float bottomRightY, float bottomRightZ, float topRightX, float topRightY, float topRightZ, float topLeftX, float topLeftY, float topLeftZ, float bottomLeftPercentX = 0.0f, float bottomLeftPercentY = 0.0f, float bottomRightPercentX = 1.0f, float bottomRightPercentY = 0.0f, float topRightPercentX = 1.0f, float topRightPercentY = 1.0f, float topLeftPercentX = 0.0f, float topLeftPercentY = 1.0f);
		//! Calculates the center of all triangles in model
		void		Calculate_Centers();
		//! Calculates the normals of all triangles in model
		void		Calculate_Normals();
		//! Calculates the radius of the model based on farthest point
		void		Calculate_Radius();
		//! Loads model from file - initialized at base point (0, 0, 0), transparency set to 0.0f, mass set to 1.0f
		bool		File_Load(char *filename);
		//! Saves model to file
		bool		File_Save(char *filename);
		//! Get mMass value
		float		Get_Mass();
		//! Get mTotalTriangles value
		int			Get_Total_Triangles();
		//! Get mTotalPoints value
		int			Get_Total_Points();
		//! Get mTransparency value for given triangle
		float		Get_Transparency(int iTriangle);
		//! Move model by given amount
		void		Move(float3 *amount, bool modifyPoints = true);
		//! Adds triangle to model from destModel if within 2D boundings (does not use .z value yet)
		bool		Oct_Tree(int iTriangle, CNeurosisModel *destModel, float3 *highBound, float3 *lowBound);
		//! Returns distance from plane
		float		Plane_Distance(int iTriangle, int iNormal, float3 *point);
		//! Returns -1 if behind, 1 if in front or on
		int			Plane_Orientation(int iTriangle, int iNormal, float3 *point);
		//! Get position of point
		float3	Polygon_Get_Position(int iTriangle, int iPoint);
		//! Returns -1 if behind, 1 if in front or on, 0 if none
		int			Polygon_Orientation(int iTriangle, float3 *point);
		//! Remove selected point
		int			Remove_Point(int index);
		//! Removes selected triangle
		int			Remove_Triangle(int index);
		//! Render model or specific triangle/mesh (textured/untextured) [renders all triangles by default]
		void		Render(int iTriangle = -1, int iMesh = -1, bool isTextured = true);
		//! Store model or specific mesh (returns updated array position)
		int			Render_Batch(int iArray, SRenderBatch batch[], int iMesh = -1);
		//! Resizes pPoints structure
		void		Resize_Points(int newSize);
		//! Resizes pTriangles structure
		void		Resize_Triangles(int newSize);
		//! Rotates the model based on the base position around the x axis (Pitch)
		void		Rotate_x(float amount);
		//! Rotates the model based on the base position around the y axis (Roll)
		void		Rotate_y(float amount);
		//! Rotates the model based on the base position around the z axis (Yaw)
		void		Rotate_z(float amount);
		//! Set mMass value [min 0.0f]
		void		Set_Mass(float amount);
		//! Set mTransparency value [0.0f..1.0f] for given triangle - if iTriangle not given, then all triangles set
		void		Set_Transparency(float amount, int iTriangle = -1);

		CNeurosisPoints			*pPoints;											// Point memory
		CNeurosisTriangles	*pTriangles;									// Triangle memory

		float								mRadius;											// Radius
		float3							mBase;												// Base point
		float3							mCurrent;											// Current position
		float3							mPrevious;										// Previous position
		float3							mVector;											// Movement vector

	private:
		float								mMass;												// Mass [min 0.0f]
		int									mTotalPoints;									// Total points
		int									mTotalTriangles;							// Total triangles
};

/***************************************************************************/
#endif